import config from "../../../config.json";
import Scrobbler, {
	type Artist,
	type Track,
	type Album,
	type User,
	type TopAlbums,
	type TopTracks,
	type Scrobbles,
} from "./Scrobbler";
import type {
	LFMAlbumResponse,
	LFMArtistResponse,
	LFMTopAlbumsResponse,
	LFMTopTracksResponse,
	LFMTrackResponse,
	LFMUserRecentResponse,
	LFMUserResponse,
} from "../../types/LastFM";

export default class LastFM extends Scrobbler {
	constructor() {
		super("lastfm", "https://ws.audioscrobbler.com/2.0/");
		if (!config.lastfm.apiKey) throw new Error("LastFM API key not set.");
	}

	public formatArtist(data: LFMArtistResponse): Artist {
		let listeners = 0,
			playcount = 0;

		try {
			listeners = parseInt(data.artist.stats.listeners, 10);
			playcount = parseInt(data.artist.stats.playcount, 10);
		} catch {
			// pass
		}

		return {
			source: "lastfm",
			url: data.artist.url,
			name: data.artist.name,
			bio: data.artist.bio.summary,
			images: data.artist.image.map((i) => {
				return {
					url: i["#text"],
					size: i.size,
				};
			}),
			similar: data.artist.similar.artist.map((a) => {
				return {
					name: a.name,
					url: a.url,
					images: a.image.map((i) => {
						return {
							url: i["#text"],
							size: i.size,
						};
					}),
				};
			}),
			stats: {
				listeners,
				playcount,
			},
			tags: data.artist.tags.tag.map((t) => t.name),
			topAlbums: [],
			topTracks: [],
			scrobbles: -1, // TODO: hardcoded
		};
	}

	public formatArtistTopAlbums(data: LFMTopAlbumsResponse): TopAlbums {
		return data.topalbums.album.map((a) => {
			let playcount = 0;

			try {
				playcount = parseInt(a.playcount, 10);
			} catch {
				// pass
			}
			return {
				name: a.name,
				url: a.url,
				playcount,
				listeners: -1, // not implemented in last.fm
				images: a.images.map((i) => {
					return {
						url: i["#text"],
						size: i.size,
					};
				}),
			};
		});
	}

	public formatArtistTopTracks(data: LFMTopTracksResponse): TopTracks {
		return data.toptracks.track.map((t) => {
			let playcount = 0,
				listeners = 0;

			try {
				playcount = parseInt(t.playcount, 10);
				listeners = parseInt(t.listeners, 10);
			} catch {
				// pass
			}
			return {
				name: t.name,
				url: t.url,
				playcount,
				listeners,
				images: t.images.map((i) => {
					return {
						url: i["#text"],
						size: i.size,
					};
				}),
			};
		});
	}

	public formatAlbum(data: LFMAlbumResponse): Album {
		let playcount = 0,
			listeners = 0;

		try {
			playcount = parseInt(data.album.playcount, 10);
			listeners = parseInt(data.album.listeners, 10);
		} catch {
			// pass
		}

		return {
			source: "lastfm",
			title: data.album.name,
			artist: data.album.artist,
			url: data.album.url,
			images: data.album.images.map((i) => {
				return {
					url: i["#text"],
					size: i.size,
				};
			}),
			release: new Date(0),
			stats: {
				listeners,
				playcount,
			},
			tags: data.album.tags.tag.map((t) => t.name),
			tracks: data.album.tracks.track.map((t) => {
				return {
					name: t.name,
					url: t.url,
					artist: t.artist,
					duration: t.duration,
				};
			}),
			scrobbles: -1, // TODO: hardcoded
		};
	}

	public formatTrack(data: LFMTrackResponse): Track {
		let playcount = 0,
			listeners = 0,
			duration = 0;

		try {
			playcount = parseInt(data.track.playcount, 10);
			listeners = parseInt(data.track.listeners, 10);
			listeners = parseInt(data.track.duration, 10);
		} catch {
			// pass
		}

		return {
			source: "lastfm",
			title: data.track.name,
			url: data.track.url,
			album: {
				title: data.track.album?.title || "unknown album (fuck you lastfm)",
				images: data.track.album?.image.map((i) => {
					return {
						url: i["#text"],
						size: i.size,
					};
				}) || [
					{
						size: "extralarge",
						url: "https://i.imgur.com/JQI9gyB.png",
					},
				],
				url: data.track.album?.url || "https://youtube.com/watch?v=dQw4w9WgXcQ",
			},
			artist: {
				name: data.track.artist.name,
				url: data.track.artist.url,
			},
			duration,
			release: new Date(0),
			stats: { listeners, playcount },
			summary: data.track.wiki?.summary || "no summary (fuck you lastfm)",
			tags: data.track.toptags.tag.map((t) => t.name),
			scrobbles: -1, // TODO: hardcoded
		};
	}

	public formatUser(data: LFMUserResponse): User {
		let tracks = 0,
			albums = 0,
			artists = 0,
			total = 0;

		try {
			tracks = parseInt(data.user.track_count, 10);
			albums = parseInt(data.user.album_count, 10);
			artists = parseInt(data.user.artist_count, 10);
			total = parseInt(data.user.playcount, 10);
		} catch {
			// pass
		}

		return {
			source: "lastfm",
			name: data.user.name,
			avatar: data.user.image[data.user.image.length - 1]["#text"] || "",
			recent: [], // TODO
			scrobbles: {
				total,
				tracks,
				albums,
				artists,
			},
			url: data.user.url,
		};
	}

	public formatUserRecent(data: LFMUserRecentResponse): User["recent"] {
		return data.recenttracks.track.map((t) => {
			return {
				name: t.name,
				artist: {
					name: t.artist["#text"],
					url: `https://www.last.fm/music/${encodeURIComponent(t.artist["#text"])}`, // HACK: might not work?
				},
				duration: -1, // not implemented in lfm
				url: t.url,
				current: !!t["@attr"]?.nowplaying,
			};
		});
	}

	public formatArtistScrobbles(data: LFMArtistResponse): Scrobbles {
		let count = -1;
		try {
			count = parseInt(data.artist.stats.userplaycount || "-1", 10);
		} catch {
			// pass
		}
		return {
			type: "artist",
			name: data.artist.name,
			count,
		};
	}

	public formatAlbumScrobbles(data: LFMAlbumResponse): Scrobbles {
		let count = -1;
		try {
			count = parseInt(data.album.userplaycount || "-1", 10);
		} catch {
			// pass
		}
		return {
			type: "album",
			name: data.album.name,
			artist: data.album.artist,
			count,
		};
	}

	public formatTrackScrobbles(data: LFMTrackResponse): Scrobbles {
		let count = -1;
		try {
			count = parseInt(data.track.userplaycount || "-1", 10);
		} catch {
			// pass
		}
		return {
			type: "track",
			name: data.track.name,
			artist: data.track.artist.name,
			count,
		};
	}

	public async getArtist(name: string): Promise<Artist> {
		const artist = await this.getOrFetch<Artist>(
			`?method=artist.getinfo&artist=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"artists",
		);
		const topAlbums = await this.getOrFetch<TopAlbums>(
			`?method=artist.gettopalbums&artist=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"artists",
		);
		const topTracks = await this.getOrFetch<TopAlbums>(
			`?method=artist.gettoptracks&artist=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"artists",
		);

		return { ...artist, topAlbums, topTracks };
	}

	public async getAlbum(artist: string, name: string): Promise<Album> {
		return await this.getOrFetch<Album>(
			`?method=album.getinfo&artist=${encodeURIComponent(artist)}&album=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"albums",
		);
	}

	public async getTrack(artist: string, name: string): Promise<Track> {
		return await this.getOrFetch<Track>(
			`?method=track.getinfo&artist=${encodeURIComponent(artist)}&track=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"tracks",
		);
	}

	public async getUser(name: string): Promise<User> {
		const user = await this.getOrFetch<User>(
			`?method=user.getinfo&user=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"users",
		);
		const recent = await this.getOrFetch<User["recent"]>(
			`?method=user.getrecenttracks&user=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"userRecent",
		);
		return { ...user, recent };
	}

	public async getUserRecent(name: string): Promise<User["recent"]> {
		return await this.getOrFetch<User["recent"]>(
			`?method=user.getrecenttracks&user=${encodeURIComponent(name)}&api_key=${config.lastfm.apiKey}&format=json`,
			"userRecent",
		);
	}

	public async getArtistScrobbles(
		user: string,
		artist: string,
	): Promise<Scrobbles> {
		return await this.getOrFetch<Scrobbles>(
			`?method=artist.getinfo&username=${encodeURIComponent(user)}&artist=${encodeURIComponent(artist)}&api_key=${config.lastfm.apiKey}&format=json`,
			"artistScrobbles",
		);
	}

	public async getAlbumScrobbles(
		user: string,
		artist: string,
		album: string,
	): Promise<Scrobbles> {
		return await this.getOrFetch<Scrobbles>(
			`?method=album.getinfo&username=${encodeURIComponent(user)}&artist=${encodeURIComponent(artist)}&album=${encodeURIComponent(album)}&api_key=${config.lastfm.apiKey}&format=json`,
			"albumScrobbles",
		);
	}

	public async getTrackScrobbles(
		user: string,
		artist: string,
		track: string,
	): Promise<Scrobbles> {
		return await this.getOrFetch<Scrobbles>(
			`?method=track.getinfo&username=${encodeURIComponent(user)}&artist=${encodeURIComponent(artist)}&track=${encodeURIComponent(track)}&api_key=${config.lastfm.apiKey}&format=json`,
			"trackScrobbles",
		);
	}

	public async searchArtist(query: string): Promise<Artist[]> {
		return [];
	}

	public async searchAlbum(query: string): Promise<Album[]> {
		return [];
	}

	public async searchTrack(query: string): Promise<Track[]> {
		return [];
	}

	public async searchUser(query: string): Promise<User[]> {
		return [];
	}
}
