import { Collection } from "discord.js";

export type Source = "unknown" | "lastfm" | "musicbrainz";
export type ImageSize = "small" | "medium" | "large" | "extralarge";

export interface Stats {
	listeners: number;
	playcount: number;
}

export interface ArtistVague {
	name: string;
	url: string;
}

export interface Image {
	url: string;
	size: ImageSize;
}

export type TopAlbums = {
	name: string;
	url: string;
	images: Image[];
	listeners: number;
}[];

export type TopTracks = {
	name: string;
	url: string;
	images: Image[];
	listeners: number;
}[];

export interface Artist {
	source: Source;
	name: string;
	images: Image[];
	url: string;
	stats: Stats;
	similar: {
		name: string;
		images: Image[];
		url: string;
	}[];
	tags: string[];
	bio: string;
	topAlbums: TopAlbums;
	topTracks: TopTracks;
	scrobbles: number;
}

export interface Track {
	source: Source;
	title: string;
	url: string;
	album: {
		url: string;
		title: string;
		images: Image[];
	};
	stats: Stats;
	artist: ArtistVague;
	tags: string[];
	summary: string;
	release: Date;
	duration: number;
	scrobbles: number;
}

export interface Album {
	source: Source;
	title: string;
	url: string;
	artist: string;
	images: Image[];
	release: Date;
	stats: Stats;
	tags: string[];
	tracks: {
		name: string;
		duration: number;
		url: string;
		artist: ArtistVague;
	}[];
	scrobbles: number;
}

export interface User {
	source: Source;
	name: string;
	url: string;
	avatar: string;
	scrobbles: {
		total: number;
		tracks: number;
		artists: number;
		albums: number;
	};
	recent: {
		name: string;
		duration: number;
		url: string;
		artist: ArtistVague;
		current: boolean;
	}[];
}

export interface Scrobbles {
	type: "artist" | "album" | "track";
	name: string;
	artist?: string;
	count: number;
}

export default abstract class Scrobbler {
	public source: Source;
	protected baseUrl: string;
	/**
	 * A cache based on query parameters. Holds data for 3 minutes.
	 * "cached" is a UNIX timestamp.
	 */
	protected cache = {
		artists: {
			data: new Collection<string, { when: number; data: Artist }>(),
			topAlbums: new Collection<string, { when: number; data: TopAlbums }>(),
			topTracks: new Collection<string, { when: number; data: TopTracks }>(),
		},
		tracks: new Collection<string, { when: number; data: Track }>(),
		albums: new Collection<string, { when: number; data: Album }>(),
		users: new Collection<string, { when: number; data: User }>(),
		userRecent: new Collection<
			string,
			{ when: number; data: User["recent"] }
		>(),
		artistScrobbles: new Collection<
			string,
			{ when: number; data: Scrobbles }
		>(),
		trackScrobbles: new Collection<string, { when: number; data: Scrobbles }>(),
		albumScrobbles: new Collection<string, { when: number; data: Scrobbles }>(),
	};

	constructor(source: Source, baseUrl: string) {
		this.source = source;
		this.baseUrl = `${baseUrl.replace(/\/+$/, "")}/`;
	}

	protected getCache(
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	) {
		if (type === "topTracks" || type === "topAlbums")
			return this.cache.artists[type];
		else return this.cache[type];
	}

	private cacheTimes = {
		artists: 1000 * 60 * 10,
		tracks: 1000 * 60 * 10,
		albums: 1000 * 60 * 10,
		users: 1000 * 60,
		userRecent: 1000 * 10,
		topAlbums: 1000 * 60 * 5,
		topTracks: 1000 * 60 * 5,
		artistScrobbles: 1000 * 60,
		trackScrobbles: 1000 * 60,
		albumScrobbles: 1000 * 60,
	};

	protected async getOrFetch<T extends Artist>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends TopAlbums>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends TopTracks>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends Album>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends Track>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends User>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends User["recent"]>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T extends Scrobbles>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T>;
	protected async getOrFetch<T>(
		path: string,
		type: keyof typeof this.cache | "topAlbums" | "topTracks",
	): Promise<T> {
		const cache = this.getCache(type) as Collection<
			string,
			{ when: number; data: T }
		>;
		const cached = cache.get(path);
		const timeInvalid = (cached?.when || 0) + this.cacheTimes[type];

		if (cached && Date.now() < timeInvalid) {
			console.log(`cache hit: ${type}`);
			return cached.data as T;
		} else {
			if (cached && Date.now() > timeInvalid) cache.delete(path);
			const response = await fetch(`${this.baseUrl}${path}`);
			if (response.status !== 200)
				throw new Error(
					`Error fetching ${path}: ${response.status} ${response.statusText}`,
				);
			const data = await response.json();

			switch (type) {
				case "artists": {
					const formatted = this.formatArtist(data);
					cache.set(path, { when: Date.now().valueOf(), data: formatted as T });
					return formatted as T;
				}

				case "topAlbums": {
					const formatted = this.formatArtistTopAlbums(data);
					cache.set(path, { when: Date.now().valueOf(), data: formatted as T });
					return formatted as T;
				}

				case "topTracks": {
					const formatted = this.formatArtistTopTracks(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}

				case "albums": {
					const formatted = this.formatAlbum(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}

				case "tracks": {
					const formatted = this.formatTrack(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}

				case "users": {
					const formatted = this.formatUser(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}

				case "userRecent": {
					const formatted = this.formatUserRecent(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}

				case "artistScrobbles": {
					const formatted = this.formatArtistScrobbles(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}
				case "trackScrobbles": {
					const formatted = this.formatTrackScrobbles(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}
				case "albumScrobbles": {
					const formatted = this.formatAlbumScrobbles(data);
					cache.set(path, { when: Date.now(), data: formatted as T });
					return formatted as T;
				}

				default: {
					break;
				}
			}

			throw new Error("Wrong cache hit.");
		}
	}

	public abstract formatArtist(data: any): Artist;
	public abstract formatArtistTopAlbums(data: any): TopAlbums;
	public abstract formatArtistTopTracks(data: any): TopTracks;
	public abstract formatAlbum(data: any): Album;
	public abstract formatTrack(data: any): Track;
	public abstract formatArtistScrobbles(data: any): Scrobbles;
	public abstract formatAlbumScrobbles(data: any): Scrobbles;
	public abstract formatTrackScrobbles(data: any): Scrobbles;
	public abstract formatUser(data: any): User;
	public abstract formatUserRecent(data: any): User["recent"];

	public abstract getArtist(name: string): Promise<Artist>;
	public abstract getAlbum(artist: string, name: string): Promise<Album>;
	public abstract getTrack(artist: string, name: string): Promise<Track>;
	public abstract getArtistScrobbles(
		user: string,
		name: string,
	): Promise<Scrobbles>;
	public abstract getAlbumScrobbles(
		user: string,
		artist: string,
		name: string,
	): Promise<Scrobbles>;
	public abstract getTrackScrobbles(
		user: string,
		artist: string,
		name: string,
	): Promise<Scrobbles>;
	public abstract getUser(name: string): Promise<User>;
	public abstract getUserRecent(name: string): Promise<User["recent"]>;

	public abstract searchArtist(query: string): Promise<Artist[]>;
	public abstract searchAlbum(query: string): Promise<Album[]>;
	public abstract searchTrack(query: string): Promise<Track[]>;
	public abstract searchUser(query: string): Promise<User[]>;
}
