import {
	Client,
	Collection,
	Events,
	MessageFlags,
	type ClientOptions,
} from "discord.js";
import { join } from "node:path";
import { readdirSync } from "node:fs";
import type CakeCommand from "./CakeCommand";
import CakeLogger from "./CakeLogger";
import CakeAL from "./CakeAL";
import LastFM from "./music/LastFMMapper";
import type CakeChatInputCommandInteraction from "./CakeChatInputCommandInteraction";

export class CakeClient<Ready extends boolean = boolean> extends Client<Ready> {
	logger: CakeLogger = new CakeLogger();
	commands: Collection<string, CakeCommand>;
	anilist: CakeAL = new CakeAL();
	lastfm: LastFM = new LastFM();

	/**
	 * Lol.
	 * @returns CakeClient
	 */
	constructor(options: ClientOptions) {
		super(options);
		this.commands = new Collection();

		const foldersPath = join(process.cwd(), "src", "commands");
		const commandFolders = readdirSync(foldersPath);
		for (const folder of commandFolders) {
			const commandsPath = join(foldersPath, folder);
			const commandFiles = readdirSync(commandsPath).filter((file) =>
				file.endsWith(".ts")
			);
			for (const file of commandFiles) {
				const filePath = join(commandsPath, file);
				import(filePath).then((object?: { default?: CakeCommand }) => {
					// Set a new item in the Collection with the key as the command name and the value as the exported module
					if (object?.default?.data) {
						const command = object.default;
						this.commands.set(command.data.name, command);
						console.log(
							`[INFO] Added ${command.data.name} to the commands.`
						);
					} else {
						console.log(
							`[WARNING] The command at ${filePath} is missing a required "data" or "execute" property.`
						);
					}
				});
			}
		}

		this.on(Events.InteractionCreate, async (iact) => {
			if (!iact.isChatInputCommand()) return;
			const interaction = iact as CakeChatInputCommandInteraction;
			const command = this.commands.get(
				interaction.commandName
			);
			if (!command) {
				console.error(
					`No command matching ${interaction.commandName} was found.`
				);
				return;
			}
			try {
				await command.execute(interaction);
			} catch (error) {
				console.error(error);
				if (interaction.replied || interaction.deferred) {
					await interaction.followUp({
						content:
							"There was an error while executing this command!",
						flags: MessageFlags.Ephemeral,
					});
				} else {
					await interaction.reply({
						content:
							"There was an error while executing this command!",
						flags: MessageFlags.Ephemeral,
					});
				}
			}
		});
	}
}
