import type {
	GetMediaResponse,
	GetUserResponse,
	GetStaffResponse,
	GetStudioResponse,
	GetUserRecentsResponse,
} from "./anilist/Get";
import type {
	SearchMediaResponse,
	SearchUserResponse,
	SearchStaffResponse,
	SearchStudioResponse,
} from "./anilist/Search";

export enum MediaTypes {
	MEDIA = "MEDIA",
	USER = "USER",
	STAFF = "STAFF",
	STUDIO = "STUDIO",
}

export const UserColors = {
	blue: "#3db3f2",
	purple: "#bf62ff",
	pink: "#fc9cd5",
	orange: "#ef8719",
	red: "#e13333",
	green: "#4cc950",
	gray: "#667a93",
};

class CakeAL {
	private static QUERY_SEARCH = {
		MEDIA:
			"query SearchMedia($search:String){Page{media(search:$search){id title{romaji english}description averageScore volumes siteUrl chapters episodes bannerImage coverImage{extraLarge color}endDate{day month year}startDate{day month year}status externalLinks{site url type}popularity tags { id name rank } rankings{rank allTime context format season type year}format studios{edges{isMain node{name}}}season seasonYear genres}}}",
		USER: "query SearchUser($search:String){Page{users(search:$search){id name}}}",
		STAFF:
			"query SearchStaff($search: String) {Page {staff(search: $search) {id name {full} favourites description image {large} staffMedia(sort:POPULARITY_DESC){edges{node{id title{romaji english}favourites siteUrl}staffRole}} siteUrl}}}",
		STUDIO:
			"query SearchStudio($search: String) {Page {studios(search: $search) {id name siteUrl favourites media(sort:FAVOURITES_DESC){edges{isMainStudio node{id title{romaji english}favourites siteUrl}}}}}}",
	};
	private static QUERY_GET = {
		MEDIA:
			"query GetMedia($id:Int){Media(id:$id){id title{romaji english}description averageScore volumes siteUrl chapters episodes bannerImage coverImage{extraLarge color}endDate{day month year}startDate{day month year}status externalLinks{site url type}popularity tags { id name rank } rankings{rank allTime context format season type year}format studios{edges{isMain node{name}}}season seasonYear genres}}",
		USER: "query GetUser($id:Int){User(id:$id){id name avatar{large}bannerImage createdAt statistics{anime{count episodesWatched meanScore}manga{chaptersRead count meanScore}}siteUrl options{profileColor}}}",
		USER_BY_NAME:
			"query GetUserByName($name:String){User(name:$name){id name avatar{large}bannerImage createdAt statistics{anime{count episodesWatched meanScore}manga{chaptersRead count meanScore}}siteUrl options{profileColor}}}",
		USER_RECENT:
			"query GetUserRecents($id:Int,$limit:Int) { Page(perPage: $limit) { activities(userId: $id, sort: ID_DESC, type: MEDIA_LIST) { ... on ListActivity { id siteUrl status progress media { id title {userPreferred} siteUrl format coverImage{color extraLarge}description episodes chapters}}}}}",
		STAFF:
			"query GetStaff($id:Int){Staff(id:$id){id name{full}favourites description image{large}staffMedia(sort:POPULARITY_DESC){edges{node{id title{romaji english}favourites siteUrl}staffRole}}siteUrl}}",
		STUDIO:
			"query GetStudio($id:Int){Studio(id:$id){id name siteUrl favourites media(sort:FAVOURITES_DESC){edges{isMainStudio node{id title{romaji english}favourites siteUrl}}}}}",
	};

	private static async fetchAL(
		query: string,
		variables: { [key: string]: string | number },
	) {
		const url = "https://graphql.anilist.co";
		const options = {
			method: "POST",
			headers: {
				"Content-Type": "application/json",
				Accept: "application/json",
			},
			body: JSON.stringify({ query, variables }),
		};

		const response = await fetch(url, options);
		if (!response.ok) {
			throw Error(
				`Failed to fetch AL (${response.status}, ${response.statusText})`,
			);
		}

		const data = await response.json();
		return data;
	}

	public async search(
		term: string,
		type: MediaTypes,
	): Promise<
		| SearchMediaResponse
		| SearchUserResponse
		| SearchStaffResponse
		| SearchStudioResponse
		| { error: "Unexpected error." }
	> {
		const query = CakeAL.QUERY_SEARCH[type];
		const variables = {
			search: term,
		};

		switch (type) {
			case MediaTypes.MEDIA: {
				const mediaData = (await CakeAL.fetchAL(query, variables)) as {
					data: {
						Page: SearchMediaResponse;
					};
				};
				return mediaData.data.Page;
			}

			case MediaTypes.USER: {
				const userData = (await CakeAL.fetchAL(query, variables)) as {
					data: {
						Page: SearchUserResponse;
					};
				};
				return userData.data.Page;
			}

			case MediaTypes.STAFF: {
				const staffData = (await CakeAL.fetchAL(query, variables)) as {
					data: {
						Page: SearchStaffResponse;
					};
				};
				return staffData.data.Page;
			}

			case MediaTypes.STUDIO: {
				const studioData = (await CakeAL.fetchAL(query, variables)) as {
					data: {
						Page: SearchStudioResponse;
					};
				};
				return studioData.data.Page;
			}

			default:
				return { error: "Unexpected error." };
		}
	}

	public async getMedia(id: string | number) {
		let _id: number;
		if (typeof id === "number") {
			_id = id;
		} else {
			try {
				_id = Number.parseInt(id, 10);
			} catch {
				throw Error(
					"how the hell did you manage to do this (id is not a number)",
				);
			}
		}

		const query = CakeAL.QUERY_GET.MEDIA;
		const variables = { id: _id };

		const data = (await CakeAL.fetchAL(query, variables)) as {
			data: GetMediaResponse;
		};

		return data.data;
	}

	public async getUser(id: string | number) {
		let _id: number;
		if (typeof id === "number") {
			_id = id;
		} else {
			try {
				_id = Number.parseInt(id, 10);
			} catch {
				throw Error(
					"how the hell did you manage to do this (id is not a number)",
				);
			}
		}

		const query = CakeAL.QUERY_GET.USER;
		const variables = { id };

		const data = (await CakeAL.fetchAL(query, variables)) as {
			data: GetUserResponse;
		};

		return data.data;
	}

	public async getUserByName(name: string) {
		const query = CakeAL.QUERY_GET.USER_BY_NAME;
		const variables = { name };

		const data = (await CakeAL.fetchAL(query, variables)) as {
			data: GetUserResponse;
		};

		return data.data;
	}

	public async getUserRecentsById(id: number, limit: number) {
		const query = CakeAL.QUERY_GET.USER_RECENT;
		const variables = { id, limit };

		const data = (await CakeAL.fetchAL(query, variables)) as {
			data: GetUserRecentsResponse;
		};

		return data.data;
	}

	public async getStaff(id: string | number) {
		let _id: number;
		if (typeof id === "number") {
			_id = id;
		} else {
			try {
				_id = Number.parseInt(id, 10);
			} catch {
				throw Error(
					"how the hell did you manage to do this (id is not a number)",
				);
			}
		}

		const query = CakeAL.QUERY_GET.STAFF;
		const variables = { id };

		const data = (await CakeAL.fetchAL(query, variables)) as {
			data: GetStaffResponse;
		};

		return data.data;
	}

	public async getStudio(id: string | number) {
		let _id: number;
		if (typeof id === "number") {
			_id = id;
		} else {
			try {
				_id = Number.parseInt(id, 10);
			} catch {
				throw Error(
					"how the hell did you manage to do this (id is not a number)",
				);
			}
		}

		const query = CakeAL.QUERY_GET.STUDIO;
		const variables = { id };

		const data = (await CakeAL.fetchAL(query, variables)) as {
			data: GetStudioResponse;
		};

		return data.data;
	}
}

export default CakeAL;
